/*
 *  Native support for the I/O-Warrior
 *
 *  Copyright (c) 2003-2004  Code Mercenaries GmbH
 *  written by Christian Lucht <lucht@codemercs.com>
 *
 *  based on
 *  usb-skeleton.c by Greg Kroah-Hartman  <greg@kroah.com>
 *  brlvger.c by Stephane Dalton  <sdalton@videotron.ca>
 *           and Stphane Doyon   <s.doyon@videotron.ca>
 *
 *  ChangeLog:
 *		v0.3.1	- bug fix in callback function for equal values on interface 0
 *      v0.3	- changes due to usb core changes with usb_register_dev()
 *      v0.2	- access to special function endpoints implemented
 *      v0.1	- Initial release, simple I/O only
 */


#include <linux/module.h>
#include <linux/usb.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include "iowarrior.h"

/* Use our own dbg macro */
#undef dbg
#define dbg( format, arg... ) do { if( debug ) printk( KERN_DEBUG __FILE__ ": " format "\n" , ## arg ); } while ( 0 )


MODULE_AUTHOR( DRIVER_AUTHOR );
MODULE_DESCRIPTION( DRIVER_DESC );
MODULE_LICENSE( "GPL" );

/* Module paramaters */
static int debug = 0;
MODULE_PARM( debug, "i" );
MODULE_PARM_DESC( debug, "Debug enabled or not" );


/*---------------------------*/
/* local function prototypes */
/*---------------------------*/
static int iowarrior_probe( struct usb_interface *interface, const struct usb_device_id *id );
static void iowarrior_disconnect( struct usb_interface *interface );

static ssize_t iowarrior_read( struct file *file, char *buffer, size_t count, loff_t *ppos );
static int iowarrior_ioctl( struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg );
static int iowarrior_open( struct inode *inode, struct file *file );
static int iowarrior_release( struct inode *inode, struct file *file );
static unsigned iowarrior_poll( struct file *file, poll_table *wait );

static void iowarrior_callback( struct urb *urb, struct pt_regs *regs );

/*--------------*/
/*     data     */
/*--------------*/

/* Structure to hold all of our device specific stuff */
struct iowarrior {
	struct semaphore				sem;					/* locks this structure */
	struct usb_device				* udev;					/* save off the usb device pointer */
	struct usb_interface			* interface;			/* the interface for this device */
	unsigned char					minor;					/* the starting minor number for this device */

	unsigned char					* ctrl_out_buffer;		/* the buffer to send data */
	__u8							ctrl_out_endpointAddr;	/* the address of the interrupt out endpoint */

	unsigned char					* int_in_buffer;		/* the buffer to read data */
	struct usb_endpoint_descriptor	* int_in_endpoint;
	struct urb						* int_in_urb;			/* the urb used to transmit read data */

	unsigned char					serial_number;			/* to detect lost packages */

	unsigned char					* read_queue;			/* size is MAX_INTERRUPT_BUFFER * packet size */
	wait_queue_head_t				read_wait;
	atomic_t						read_idx;
	atomic_t						intr_idx;
	spinlock_t						intr_idx_lock;			/* protects intr_idx */
	atomic_t						overflow_flag;			/* signals an index 'rollover' */

	int								present;				/* if the device is not disconnected */
	int								opened;
	struct semaphore				open_sem;				/* protects ->opened */
};


/*--------------*/
/*    globals   */
/*--------------*/
/* prevent races between open() and disconnect() */
static DECLARE_MUTEX( disconnect_sem );

/*
 *  USB spec identifies 5 second timeouts.
 */
#define GET_TIMEOUT 5
#define USB_REQ_GET_REPORT  0x01
static int usb_get_report( struct usb_device *dev, struct usb_host_interface *inter, unsigned char type, unsigned char id, void *buf, int size)
{
	return usb_control_msg( dev, usb_rcvctrlpipe( dev, 0 ),
	                        USB_REQ_GET_REPORT, USB_DIR_IN | USB_TYPE_CLASS | USB_RECIP_INTERFACE,
	                        (type << 8) + id, inter->desc.bInterfaceNumber, buf, size, HZ * GET_TIMEOUT);
}


#define USB_REQ_SET_REPORT	0x09
static int usb_set_report( struct usb_interface *intf, unsigned char type, unsigned char id, void *buf, int size )
{
	return usb_control_msg( interface_to_usbdev( intf ), usb_sndctrlpipe( interface_to_usbdev( intf ), 0 ),
	                        USB_REQ_SET_REPORT, USB_TYPE_CLASS | USB_RECIP_INTERFACE,
	                        ( type << 8 ) + id, intf->altsetting[0].desc.bInterfaceNumber, buf, size, HZ );
}


/*---------------------*/
/* driver registration */
/*---------------------*/
/* table of devices that work with this driver */
struct usb_device_id iowarrior_ids[] = {
	{ USB_DEVICE(USB_VENDOR_ID_CODEMERCS, USB_DEVICE_ID_CODEMERCS_IOW40) },
	{ USB_DEVICE(USB_VENDOR_ID_CODEMERCS, USB_DEVICE_ID_CODEMERCS_IOW24) },
	{}				/* Terminating entry */
};
MODULE_DEVICE_TABLE( usb, iowarrior_ids );


/*
 * File operations needed when we register this driver.
 * This assumes that this driver NEEDS file operations,
 * of course, which means that the driver is expected
 * to have a node in the /dev directory. If the USB
 * device were for a network interface then the driver
 * would use "struct net_driver" instead, and a serial
 * device would use "struct tty_driver".
 */
static struct file_operations iowarrior_fops = {
	/*
	 * The owner field is part of the module-locking
	 * mechanism. The idea is that the kernel knows
	 * which module to increment the use-counter of
	 * BEFORE it calls the device's open() function.
	 * This also means that the kernel can decrement
	 * the use-counter again before calling release()
	 * or should the open() function fail.
	 */
	.owner =	THIS_MODULE,

	.read =		iowarrior_read,
	.ioctl =	iowarrior_ioctl,
	.open =		iowarrior_open,
	.release =	iowarrior_release,
	.poll =		iowarrior_poll
};

/*
 * usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with devfs and the driver core
 */
static struct usb_class_driver iowarrior_class = {
	.name =		"usb/iowarrior%d",
	.fops =		&iowarrior_fops,
	.mode =		S_IFCHR | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH,
	.minor_base =	IOWARRIOR_MINOR_BASE,
};


/* usb specific object needed to register this driver with the usb subsystem */
static struct usb_driver iowarrior_driver = {
	.owner =	THIS_MODULE,
	.name =		"iowarrior",
	.probe = 	iowarrior_probe,
	.disconnect =	iowarrior_disconnect,
	.id_table =	iowarrior_ids,
};


/**
 *	iowarrior_init
 */
static int __init iowarrior_init( void )
{

	int result;

	/* register this driver with the USB subsystem */
	result = usb_register( &iowarrior_driver );
	if( result < 0 )
	{
		err( "usb_register failed for the "__FILE__" driver. Error number %d", result );
		return -1;
	}

	info( DRIVER_DESC " " DRIVER_VERSION );
	return 0;
}


/**
 *	iowarrior_exit
 */
static void __exit iowarrior_exit( void )
{
	/* deregister this driver with the USB subsystem */
	usb_deregister( &iowarrior_driver );
}


module_init( iowarrior_init );
module_exit( iowarrior_exit );


/**
 *	iowarrior_delete
 */
static inline void iowarrior_delete( struct iowarrior *dev )
{
	kfree( dev->int_in_buffer );

	if( dev->ctrl_out_buffer != NULL )
		kfree( dev->ctrl_out_buffer );

	usb_free_urb( dev->int_in_urb );

	kfree( dev->read_queue );

	kfree( dev );
}


/*---------------------------------*/
/*  probe and disconnect functions */
/*---------------------------------*/
/**
 *	iowarrior_probe
 *
 *	Called by the usb core when a new device is connected that it thinks
 *	this driver might be interested in.
 */
static int iowarrior_probe( struct usb_interface *interface, const struct usb_device_id *id )
{
	struct usb_device *udev = interface_to_usbdev( interface );
	struct iowarrior *dev = NULL;
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	int i;
	int retval = -ENOMEM;

	/* See if the device offered us matches what we can accept */
	if(( udev->descriptor.idVendor != USB_VENDOR_ID_CODEMERCS ) ||
	  (( udev->descriptor.idProduct != USB_DEVICE_ID_CODEMERCS_IOW40 ) &&
	   ( udev->descriptor.idProduct != USB_DEVICE_ID_CODEMERCS_IOW24 )))
	{
		return -ENODEV;
	}

	/* allocate memory for our device state and intialize it */
	dev = kmalloc( sizeof( struct iowarrior ), GFP_KERNEL );
	if( dev == NULL )
	{
		err( "Out of memory" );
		return retval;
	}

	memset( dev, 0x00, sizeof( *dev ));

	init_MUTEX( &dev->sem );

	dev->int_in_buffer = NULL;
	dev->int_in_endpoint = NULL;
	dev->int_in_urb = NULL;

	atomic_set( &dev->intr_idx, 0 );
	atomic_set( &dev->read_idx, 0 );
	spin_lock_init( &dev->intr_idx_lock );
	atomic_set( &dev->overflow_flag, 0 );
	init_waitqueue_head( &dev->read_wait );
	/* opened is memset'ed to 0 */
	init_MUTEX( &dev->open_sem );

	dev->udev = udev;
	dev->interface = interface;

	iface_desc = &interface->altsetting[0];

	/* set up the endpoint information */
	for( i = 0; i < iface_desc->desc.bNumEndpoints; ++i )
	{
		endpoint = &iface_desc->endpoint[i].desc;

		if((( endpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK ) == USB_DIR_IN ) &&
		   (( endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK ) == USB_ENDPOINT_XFER_INT ))
		{
			dev->int_in_endpoint = endpoint;
		}
	}

	if( dev->int_in_endpoint == NULL )
	{
		err( "interrupt in  endpoint not found" );
		goto error;
	}

	dev->int_in_buffer = kmalloc( dev->int_in_endpoint->wMaxPacketSize, GFP_KERNEL );
	if( !dev->int_in_buffer )
	{
		err( "Couldn't allocate int_in_buffer" );
		goto error;
	}

	/* +1 is for the time stamp */
	dev->read_queue = kmalloc((( dev->int_in_endpoint->wMaxPacketSize + 1 ) * MAX_INTERRUPT_BUFFER ), GFP_KERNEL );
	if( !dev->read_queue )
	{
		err( "Couldn't allocate read_queue" );
		goto error;
	}

	/* allow device read and ioctl */
	dev->present = 1;

	/* we can register the device now, as it is ready */
	usb_set_intfdata( interface, dev );

	retval = usb_register_dev( interface, &iowarrior_class );
	if( retval )
	{
		/* something prevented us from registering this driver */
		err( "Not able to get a minor for this device." );
		usb_set_intfdata( interface, NULL );
		goto error;
	}

  	dev->minor = interface->minor;

	/* let the user know what node this device is now attached to */
	info( "I/O-Warrior device now attached to iowarrior%d", dev->minor - IOWARRIOR_MINOR_BASE );
	return retval;

error:
	iowarrior_delete( dev );
	return retval;
}


/**
 *	iowarrior_disconnect
 *
 *	Called by the usb core when the device is removed from the system.
 */
static void iowarrior_disconnect( struct usb_interface *interface )
{
	struct iowarrior *dev;
	int minor;

	/* prevent races with open() */
	down( &disconnect_sem );

	dev = usb_get_intfdata( interface );
	usb_set_intfdata( interface, NULL );

	down( &dev->sem );

	minor = dev->minor;

	/* give back our minor */
	usb_deregister_dev( interface, &iowarrior_class );

	/* prevent device read, write and ioctl */
	dev->present = 0;

	up( &dev->sem );

	/* if the device is opened, then we clean up right now */
	if( !dev->opened )
	{
		iowarrior_delete( dev );
	}
	wake_up_interruptible( &dev->read_wait );

	up( &disconnect_sem );

	info( "I/O-Warror #%d now disconnected", minor );
}


/*---------------------*/
/* fops implementation */
/*---------------------*/

static int read_index( struct iowarrior *dev )
{
	int intr_idx, read_idx;

	read_idx = atomic_read( &dev->read_idx );
	intr_idx = atomic_read( &dev->intr_idx );

	return( read_idx == intr_idx ? -1 : read_idx );
}


 /**
  *  iowarrior_read
  */
static ssize_t iowarrior_read( struct file *file, char *buffer, size_t count, loff_t *ppos )
{
	struct iowarrior *dev;
	int read_idx;

	dev = (struct iowarrior *)file->private_data;

	dbg( "%s - minor %d, count = %d", __func__, dev->minor, count );

	/* read count must be packet size (+ time stamp) */
	if(( count !=  dev->int_in_endpoint->wMaxPacketSize )
	&& ( count != ( dev->int_in_endpoint->wMaxPacketSize + 1 )))
		return -EINVAL;

	/* verify that the device wasn't unplugged */
	if( dev->udev == NULL )
		return -ENODEV;

	/* repeat until no buffer overrun in callback handler occur */
	do
	{
		atomic_set( &dev->overflow_flag, 0 );
		if(( read_idx = read_index( dev )) == -1 )
		{
			/* queue emty */
			if( file->f_flags & O_NONBLOCK )
				return -EAGAIN;
			else
			{
				int r = wait_event_interruptible( dev->read_wait,
				                                 ( !dev->udev || ( read_idx = read_index( dev )) != -1 ));
				if( !dev->udev )
					return -ENOLINK;
				if( r )
					return r;
				if( read_idx == -1 )
					/* should not happen */
					return 0;
			}
		}

		int offset = read_idx * (dev->int_in_endpoint->wMaxPacketSize + 1 );
		if( copy_to_user( buffer, dev->read_queue+offset, ( dev->int_in_endpoint->wMaxPacketSize + 1 )))
			return -EFAULT;
	}
	while( atomic_read( &dev->overflow_flag ));

	read_idx = ++read_idx == MAX_INTERRUPT_BUFFER ? 0 : read_idx;
	atomic_set( &dev->read_idx, read_idx );

	return count;
}


/**
 *	iowarrior_ioctl
 */
static int iowarrior_ioctl( struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg )
{
	struct iowarrior *dev = NULL;
	dev = (struct iowarrior *)file->private_data;
	__u8 buffer[8];		/* max packet size for iowarrior */

	if( dev == NULL )
	{
		return -ENODEV;
	}

	/* lock this object */
	down (&dev->sem);

	/* verify that the device wasn't unplugged */
	if( !dev->present )
	{
		up (&dev->sem);
		return -ENODEV;
	}

	dbg( "%s - minor %d, cmd 0x%.4x, arg %ld", __func__, dev->minor, cmd, arg );

	int retval = 0;
	switch( cmd )
	{
		case IOW_WRITE:
		{
			int wrote;
			copy_from_user( (void *)buffer, (const void *)arg, dev->int_in_endpoint->wMaxPacketSize );

			// 2 = Report Type Output, 0 = Report IDs are not used
			wrote = usb_set_report( dev->interface, 2, 0, (void *)buffer, dev->int_in_endpoint->wMaxPacketSize );
			if( wrote < 0 )
				retval = wrote;

			break;
		}
		case IOW_READ:
		{
			int read;

			// 1 = Report Type Input, 0 = Report IDs are not used
			read = usb_get_report( dev->udev, dev->interface->altsetting, 1, 0, (void *)buffer, dev->int_in_endpoint->wMaxPacketSize );
			if( read < 0 )
				retval = read;
			else
				copy_to_user( (void *)arg, (void *)buffer, dev->int_in_endpoint->wMaxPacketSize );
			break;
		}
		default:
		{
			/* return that we did not understand this ioctl call */
			retval = -ENOTTY;
			break;
		}
	}

	/* unlock the device */
	up( &dev->sem );
	return retval;
}


/**
 *	iowarrior_open
 */
static int iowarrior_open( struct inode *inode, struct file *file )
{
	struct iowarrior *dev = NULL;
	struct usb_interface *interface;
	int subminor;
	int retval = 0;

	dbg( "%s", __func__ );

	subminor = iminor( inode );

	/* prevent disconnects */
	down( &disconnect_sem );

	interface = usb_find_interface( &iowarrior_driver, subminor );
	if( !interface )
	{
		err( "%s - error, can't find device for minor %d", __func__, subminor );
		retval = -ENODEV;
		goto out;
	}

	dev = usb_get_intfdata( interface );
	if (!dev) {
		retval = -ENODEV;
		goto out;
	}

	/* Only one process can open each device, no sharing. */
	retval = -EBUSY;
	if( dev->opened )
		goto out;

	dbg( "Opening iowarrior%d", dev->minor );

	/* setup interrupt handler for receiving values */
	dev->int_in_urb = usb_alloc_urb( 0, GFP_KERNEL );
	if( !dev->int_in_urb )
	{
		err( "Couldn't allocate interrupt_in_urb" );
		goto out;
	}

	usb_fill_int_urb( dev->int_in_urb, dev->udev,
	                  usb_rcvintpipe( dev->udev, dev->int_in_endpoint->bEndpointAddress ),
	                  dev->int_in_buffer, dev->int_in_endpoint->wMaxPacketSize,
	                  iowarrior_callback, dev, dev->int_in_endpoint->bInterval );


	if(( retval = usb_submit_urb( dev->int_in_urb, GFP_KERNEL )) < 0 )
	{
		err( "Error %d while submitting URB", retval );
		goto out;
	}

	/* increment our usage count for the driver */
	++dev->opened;
	/* save our object in the file's private structure */
	file->private_data = dev;
	retval = 0;

out:
	up( &disconnect_sem );
dbg( "out %s", __func__ );
	return retval;
}


/**
 *	iowarrior_release
 */
static int iowarrior_release( struct inode *inode, struct file *file )
{
	struct iowarrior *dev;
	int retval = 0;

	dev = (struct iowarrior *)file->private_data;
	if( dev == NULL )
	{
		dbg( "%s - object is NULL", __func__ );
		return -ENODEV;
	}

	dbg( "%s - minor %d", __func__, dev->minor );

	/* lock our device */
	down( &dev->sem );

	if( dev->opened <= 0 )
	{
		dbg( "%s - device not opened", __func__ );
		retval = -ENODEV;
		goto exit_not_opened;
	}

	/* shutdown any reads that might be going on */
	usb_unlink_urb( dev->int_in_urb );

	--dev->opened;

	if( !dev->present && !dev->opened )
	{
		/* the device was unplugged before the file was released */
		up( &dev->sem );
		iowarrior_delete( dev );
		return 0;
	}

exit_not_opened:
	up( &dev->sem );
	return retval;
}


static unsigned iowarrior_poll( struct file *file, poll_table *wait )
{
	struct iowarrior *dev = file->private_data;

	if( !dev->udev )
		return POLLERR | POLLHUP;

	poll_wait( file, &dev->read_wait, wait );

	if( !dev->udev )
		return POLLERR | POLLHUP;

	if( read_index( dev ) != -1 )
		return POLLIN | POLLRDNORM;

	return 0;
}


/*
 * USB callback handler for reading data
 */
static void iowarrior_callback( struct urb *urb, struct pt_regs *regs )
{
	struct iowarrior *dev = (struct iowarrior *)urb->context;
	int intr_idx;
	int read_idx;
	int aux_idx;
	int offset;
	int status;

	switch( urb->status )
	{
	case 0:
		/* success */
		break;
	case -ECONNRESET:
	case -ENOENT:
	case -ESHUTDOWN:
		/* this urb is terminated, clean up */
		dbg( "%s - urb shutting down with status: %d", __FUNCTION__, urb->status );
		return;
	default:
		dbg( "%s - nonzero urb status received: %d", __FUNCTION__, urb->status );
		goto exit;
	}

	spin_lock( &dev->intr_idx_lock );
	intr_idx = atomic_read( &dev->intr_idx );
	/* aux_idx become previous intr_idx */
	aux_idx = ( intr_idx == 0 ) ? ( MAX_INTERRUPT_BUFFER - 1 ) : ( intr_idx - 1 );
	read_idx = atomic_read( &dev->read_idx );

	/* queue is not empty and it's interface 0 */
	if(( intr_idx != read_idx ) && ( dev->interface->altsetting->desc.bInterfaceNumber == 0 ))
	{
		/* + 1 for serial number */
		offset = aux_idx * ( dev->int_in_endpoint->wMaxPacketSize + 1 );
		if( !memcmp( dev->read_queue+offset, urb->transfer_buffer, dev->int_in_endpoint->wMaxPacketSize ))
		{
			/* equal values on interface 0 will be ignored */
			spin_unlock( &dev->intr_idx_lock );
			goto exit;
		}
	}

	/* aux_idx become next intr_idx */
	aux_idx = ( intr_idx == ( MAX_INTERRUPT_BUFFER - 1 )) ? 0 : ( intr_idx + 1 );
	if( read_idx == aux_idx )
	{
		/* queue full, dropping oldest input */
		read_idx = ( ++read_idx == MAX_INTERRUPT_BUFFER ) ? 0 : read_idx;
		atomic_set( &dev->read_idx, read_idx );
		atomic_set( &dev->overflow_flag, 1 );
	}

	/* +1 for serial number */
	offset = intr_idx * ( dev->int_in_endpoint->wMaxPacketSize + 1 );
	memcpy( dev->read_queue+offset, urb->transfer_buffer, dev->int_in_endpoint->wMaxPacketSize );
	*(dev->read_queue+offset+(dev->int_in_endpoint->wMaxPacketSize )) = dev->serial_number++;

	atomic_set( &dev->intr_idx, aux_idx );
	spin_unlock( &dev->intr_idx_lock );

	wake_up_interruptible( &dev->read_wait );
	
exit:
	status = usb_submit_urb( urb, GFP_ATOMIC );
	if( status )
		err( "%s - usb_submit_urb failed with result %d", __FUNCTION__, status );

}


/**
 *	iowarrior_debug_data
 */
static inline void iowarrior_debug_data (const char *function, int size, const unsigned char *data)
{
	int i;

	if (!debug)
		return;

	printk (KERN_DEBUG __FILE__": %s - length = %d, data = ",
		function, size);
	for (i = 0; i < size; ++i)
	{
		printk ("%.2x ", data[i]);
	}
	printk ("\n");
}
